<?php
declare(strict_types=1);

namespace App\Core;

final class ErrorHandler
{
    public static function register(bool $debug, string $logFile): void
    {
        ini_set('display_errors', $debug ? '1' : '0');
        ini_set('log_errors', '1');
        ini_set('error_log', $logFile);

        set_error_handler(function (int $severity, string $message, string $file, int $line): bool {
            if (!(error_reporting() & $severity)) {
                return false;
            }
            throw new \ErrorException($message, 0, $severity, $file, $line);
        });

        set_exception_handler(function (\Throwable $e) use ($debug, $logFile): void {
            $msg = sprintf("[%s] %s in %s:%d\n%s\n\n", date('c'), $e->getMessage(), $e->getFile(), $e->getLine(), $e->getTraceAsString());
            @file_put_contents($logFile, $msg, FILE_APPEND);

            http_response_code(500);
            if ($debug) {
                echo "<h1>500 - Error</h1>";
                echo "<pre>" . htmlspecialchars($msg, ENT_QUOTES, 'UTF-8') . "</pre>";
                return;
            }

            $view = APP_PATH . '/Views/errors/500.php';
            if (is_file($view)) {
                require $view;
                return;
            }
            echo '500 - Error interno';
        });
    }
}
